####################################
# Bayesian Binary Two-Group Comparison
# Author: Michail Tsikerdekis
# Description: It compares two group binary rates and obtains their difference delta using MCMC. 
# It also tests the null hypothesis based on delta = 0.
####################################

# Clear Workspace
rm(list=ls(all=TRUE)) 
set.seed=06192014
#setwd("~/Dropbox/BayesChapter") # if source files are not found point to the directory using this command

####################################
# Book data preparation
####################################
source("generate.R")

####################################
# Data Input
# group 1 is a list containing all the interval values obtained for group 1
# group 2 is a list containing all the interval values obtained for group 2
# Priors s1prior, s2prior, f1prior, f2prior = successes + 1 and failures + 1. If both 0 that is considered a uniform
# prior with equal probabilities across the board
####################################
group1 = email$responseTime[email$team == 0]
group2 = email$responseTime[email$team == 1]

muXprior = 1
muYprior = 1
sdXprior = 1
sdYprior = 1

# Rescale:
#group2 = group2-mean(group1)
#group2 = group2/sd(group1)
#group1 = as.vector(scale(group1))

# Model parameters
n.simu <- 30000
n.burnin <- n.simu/2
n.burnin <- 0
par <- c("muX","muY","delta")

# Hypothesis testing parameters
equalvariances <- F
confint = .95 # Confidence interval
compval = 0 # The null value. For HDI in ROPE this can be virtually anything.
ropeRad = 0.2 # ROPE radius in case ROPE is used to test hypotheses

# Others options
convergencetests <- T
plotting <- T
testbayes <- T
testrope <- T

####################################
# Additional calculated variables
####################################
n1 = length(group1)
n2 = length(group2)
D <- list(group1 = group1, group2 = group2, n1 = n1, n2 = n2)

####################################
# Load or Install Required Packages
####################################
packages = c("R.utils","R2jags","MCMCpack","coda","R2OpenBUGS","plyr","ggplot2","plyr","BEST")
for (package in packages){
  if (require(package,character.only=TRUE)) {require(package,character.only=TRUE)} else {
    install.packages(package)
    library(package,character.only=TRUE) # This will ensure that the code fails if the installation failed
  }
}

####################################
# Setup JAGS model
####################################
if (!equalvariances) {
  jags.bin <- function() {
  
    # Priors
    muX ~ dnorm(0, 0.001)
    muY ~ dnorm(0, 0.001)
    sigmaX ~ dunif(0, 1000)
    sigmaY ~ dunif(0, 1000)
    tauX <- 1 / (sigmaX * sigmaX)
    tauY <- 1 / (sigmaY * sigmaY) 

    # Likelihood
    for (i in 1:n1) {
      group1[i] ~ dnorm(muX, tauX)
    }
    for (i in 1:n2) {
      group2[i] ~ dnorm(muY, tauY)
    }
    
    # Derived quantities
    delta <- muY - muX
  }
} else {
  jags.bin <- function() {
    # Priors
    muX ~ dnorm(0, 0.001)
    muY ~ dnorm(0, 0.001)
    sigmaXY ~ dunif(0, 1000)
    tauXY <- 1 / (sigmaXY * sigmaXY)
    
    # Likelihood
    for (i in 1:n1) {
      group1[i] ~ dnorm(muX, tauXY)
    }
    for (i in 1:n2) {
      group2[i] ~ dnorm(muY, tauXY)
    }
    
    # Derived quantities
    delta <- muY - muX
  }
}
write.model(jags.bin, "jags.txt")

####################################
# Building model using MCMC sampling
####################################
m.jags <- jags.model("jags.txt", data = D, n.adapt = n.burnin, quiet = TRUE, n.chains = 4)
s <- coda.samples(m.jags, c("muX","muY"), n.iter = n.simu - n.burnin, quiet = TRUE)
if (convergencetests){gelman.plot(s)}
if (convergencetests){geweke.diag(s)}
s <- coda.samples(m.jags, par, n.iter = n.simu - n.burnin, quiet = TRUE)
if (convergencetests){plot(s)}

# Converting coda-object to data frame object in order to handle chains and variables.
df = as.data.frame(as.matrix( s ))

####################################
# Plotting posterior results
####################################
if (plotting){
  # Comparing density posterior plots of two thetas
  nrecords = nrow(df)
  dfplot = data.frame(Means = c(df$muX,df$muY), Group = c(rep("Group A", nrecords),rep("Group B", nrecords)))
  cdf <- ddply(dfplot, "Group", summarise, theta.mean=mean(Means))
  print(ggplot(dfplot, aes(x=Means, fill=Group)) + geom_density(alpha = .3) + theme_minimal() +
    geom_vline(data=cdf, aes(xintercept=theta.mean,  colour=Group), linetype="dashed", size=1) +
    ggtitle("Probability Distribution for Group Means") +xlab(bquote(paste("Means"))) +
    ylab("Probability Density"))
}


####################################
# Means for Thetas and HDI intervals
####################################
meantheta1 = mean(df$muX)
meantheta2 = mean(df$muY)
hditheta1 = hdi(df$muX,.95)
hditheta2 = hdi(df$muY,.95)
cat(paste("Group 1: M = ",round(meantheta1,3),", 95% HDI [",round(hditheta1[[1]],3),", ",round(hditheta1[[2]],3),"]\n",sep=""))
cat(paste("Group 2: M = ",round(meantheta2,3),", 95% HDI [",round(hditheta2[[1]],3),", ",round(hditheta2[[2]],3),"]\n",sep=""))

####################################
# Model Comparison
####################################
# Testing Null Hypothesis using Kruschke's HDI and ROPE method
if (testrope) {
  source("H0testUsingROPE.R")
  H0testUsingROPE(df$delta,compval, ropeRad, confint)
}

